﻿/* Copyright 2017, L'Express de Toronto Inc.
Script: Move object by column
Version: 1.2
December 20, 2017
Written by Kasyan Servetsky
http://www.kasyan.ho.com.ua
e-mail: askoldich@yahoo.com

Warning: this script should be run from the 'Vertical/Horizontal.jsx'  */
//======================================================================================
var scriptName = "Cutoff rule";

if (typeof arguments == "undefined") {
	ErrorExit("This script should be run from the 'Vertical/Horizontal.jsx' script.", true);
}

var verticalRule = arguments[0],
doc, objStyleCutoffRule,
selectedItems = [],
selectedGraphicLines = [],
selectedFrames = [];

PreCheck();

//===================================== FUNCTIONS ======================================
function Main() {
	// Save & change units
	var savedHorizontalMeasurementUnits = doc.viewPreferences.horizontalMeasurementUnits;
	var savedVerticalMeasurementUnits = doc.viewPreferences.verticalMeasurementUnits;
	doc.viewPreferences.horizontalMeasurementUnits = MeasurementUnits.POINTS;
	doc.viewPreferences.verticalMeasurementUnits = MeasurementUnits.POINTS;
	
	try {
		var pageCenter,
		line = null,
		selBounds = GetSelBounds(),
		shift = (verticalRule) ? 5.638 : 10, // 5.638 pt =  0.0783 in
		page = selectedItems[0].parentPage,
		pb = page.bounds,
		pageIsLeft = (page.side == PageSideOptions.LEFT_HAND) ? true : false;
		
		if (pageIsLeft) {
			pageCenter = (pb[3] - pb[1]) / 2;
		}
		else {
			pageCenter = pb[1] + (pb[3] - pb[1]) / 2;
		}

		if (verticalRule) { // Vertical
			if ((selectedFrames.length == 1 || selectedFrames.length > 1 && VerticallyAligned()) && selectedGraphicLines.length == 0) { // With one frame selected, or all frames are vertically alined
				line = page.graphicLines.add();

				if (pageIsLeft) {
					if (selBounds.middle < pageCenter) {
						line.geometricBounds = [selBounds.top, selBounds.right + shift, selBounds.bottom, selBounds.right + shift];
					}
					else if (selBounds.middle > pageCenter) {
						line.geometricBounds = [selBounds.top, selBounds.left - shift, selBounds.bottom, selBounds.left - shift];
					}
				}
				else {
					if (selBounds.middle < pageCenter) {
						line.geometricBounds = [selBounds.top, selBounds.right + shift, selBounds.bottom, selBounds.right + shift];
					}
					else if (selBounds.middle > pageCenter) {
						line.geometricBounds = [selBounds.top, selBounds.left - shift, selBounds.bottom, selBounds.left - shift];
					}
				}
			}
			else if (selectedFrames.length == 2 && selectedGraphicLines.length == 0) { // With two frames selected
				var frameA = selectedFrames[0];
				var gbA = frameA.geometricBounds;
				var frameA_width = gbA[3] - gbA[1];
		
				var frameB = selectedFrames[1];
				var gbB = frameB.geometricBounds;				
				var frameB_width = gbB[3] - gbB[1];

				if (selBounds.width > frameA_width + frameB_width) { // two frames are located horizontally
					if (gbA[3] < gbB[1]) {
						var rightEdge = gbA[3];
						var leftEdge = gbB[1];
					}
					else if (gbB[3] < gbA[1]) {
						var rightEdge = gbB[3];
						var leftEdge = gbA[1];
					}
				
					var betweenPosition = rightEdge + ((leftEdge - rightEdge) / 2);
					line = page.graphicLines.add();
					line.geometricBounds = [selBounds.top, betweenPosition, selBounds.bottom, betweenPosition];				
				}
				else { // two frames are located vertically: one under another
					var acceptableDifference = 3;
					
					if (ApproximatelyEquals(gbA[1], gbB[1], acceptableDifference) || ApproximatelyEquals(gbA[3], gbB[3], acceptableDifference)) {
						line = page.graphicLines.add();
						
						if (pageIsLeft) {
							if (selBounds.middle < pageCenter) {
								line.geometricBounds = [selBounds.top, selBounds.right + shift, selBounds.bottom, selBounds.right + shift];
							}
							else if (selBounds.middle > pageCenter) {
								line.geometricBounds = [selBounds.top, selBounds.left - shift, selBounds.bottom, selBounds.left - shift];
							}
						}
						else {
							if (selBounds.middle < pageCenter) {
								line.geometricBounds = [selBounds.top, selBounds.right + shift, selBounds.bottom, selBounds.right + shift];
							}
							else if (selBounds.middle > pageCenter) {
								line.geometricBounds = [selBounds.top, selBounds.left - shift, selBounds.bottom, selBounds.left - shift];
							}
						}
					}
				}
			}
			else if (selectedFrames.length >= 1 && selectedGraphicLines.length == 1) { // With one or more frames and a line selected
				line = selectedGraphicLines[0];
				var lgb = line.geometricBounds;
				line.geometricBounds = [selBounds.top, lgb[1], selBounds.bottom, lgb[3]];
			}
		}
		else { // Horizontal
			if (selectedFrames.length == 1) { // With one frame selected
				line = page.graphicLines.add();
				var fgb = selectedFrames[0].geometricBounds;
				line.geometricBounds = [fgb[0] - shift, fgb[1], fgb[0] - shift, fgb[3]];
			}
			else if (selectedFrames.length == 2) { // With two frames selected
				line = page.graphicLines.add();
				line.geometricBounds = [selBounds.top - shift, selBounds.left, selBounds.top - shift, selBounds.right];
			}
		}

		if (line != null) {
			line.applyObjectStyle(objStyleCutoffRule);
		}
		else {
			alert("No line has been added.", scriptName, true);
		}

	}
	catch(err) {
		// Restore original units if something goes wrong
		doc.viewPreferences.horizontalMeasurementUnits = savedHorizontalMeasurementUnits;
		doc.viewPreferences.verticalMeasurementUnits = savedVerticalMeasurementUnits;
		//$.writeln(err.message + ", line: " + err.line);
	}

	// Restore original units
	doc.viewPreferences.horizontalMeasurementUnits = savedHorizontalMeasurementUnits;
	doc.viewPreferences.verticalMeasurementUnits = savedVerticalMeasurementUnits;
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------
function GetSelBounds() {
	var item, gb, left, right, top, bottom;
	for (var i = 0; i < selectedItems.length; i++) {
		if (selectedItems[i].constructor.name == "TextFrame" || selectedItems[i].constructor.name == "Rectangle") {
			item = selectedItems[i];
			gb = item.geometricBounds;
			
			if (left == undefined || gb[1] < left) left = gb[1];
			if (right == undefined || gb[3] > right) right = gb[3];
			if (top == undefined || gb[0] < top) top = gb[0];
			if (bottom == undefined || gb[2] > bottom) bottom = gb[2];
		}
	}
	
	var width = Math.floor((right-left)*100)/100;
	var height = Math.floor((bottom-top)*100)/100;
	var middle = left + (width/2);
	
	return { width : width, height : height, left : left, right : right, top : top, bottom : bottom, middle : middle };
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------
function VerticallyAligned() {
	var frame, gb,
	tolerance = 3,
	leftMin = selectedFrames[0].geometricBounds[1] - tolerance,
	leftMax = selectedFrames[0].geometricBounds[1] + tolerance,
	rightMin = selectedFrames[0].geometricBounds[3] - tolerance,
	rightMax = selectedFrames[0].geometricBounds[3] + tolerance,
	result = true;
	
	for (var i = 1; i < selectedFrames.length; i++) {
		frame = selectedFrames[i];
		gb = frame.geometricBounds;
		
		if (!(gb[1] >= leftMin && gb[1] <= leftMax) || !(gb[3] >= rightMin && gb[3] <= rightMax)) {
			result = false;
			break;
		}
	}
	
	return result;
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------
function ApproximatelyEquals(value1, value2, acceptableDifference) {
	return Math.abs(value1 - value2) <= acceptableDifference; 
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------
function GetSelection() {
	var sel,
	selection = app.selection;
	
	for (var i = 0; i < selection.length; i++) {
		sel = selection[i];
		
		if (sel.hasOwnProperty("geometricBounds")) {
			selectedItems.push(sel);

			if (sel.constructor.name == "GraphicLine") {
				selectedGraphicLines.push(sel);
			}
			else if (sel.constructor.name == "Rectangle" || sel.constructor.name == "TextFrame") {
				selectedFrames.push(sel);
			}
		}
	}
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------
function PreCheck() {
	if (app.documents.length == 0) ErrorExit("Please open a document and try again.", true);
	doc = app.activeDocument;
	if (doc.converted) ErrorExit("The current document has been modified by being converted from older version of InDesign. Please save the document and try again.", true);
	if (!doc.saved) ErrorExit("The current document has not been saved since it was created. Please save the document and try again.", true);
	
	GetSelection();
	if (app.selection.length == 0) ErrorExit("Nothing is selected.", true);
	if (selectedItems.length == 0) ErrorExit("Invalid selection.", true);
	if (selectedFrames.length == 0) ErrorExit("No frame is selected.", true);
	
	objStyleCutoffRule = doc.objectStyles.itemByName("Cutoff rule");
	if (!objStyleCutoffRule.isValid)  ErrorExit("Object style 'Cutoff rule' doesn't exist.", true);
	
	Main();
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------
function ErrorExit(error, icon) {
	alert(error, scriptName, icon);
	exit();
}
//--------------------------------------------------------------------------------------------------------------------------------------------------------
/* Description:

Cutoff rule vertical
The goal is automatically place a Cutoff Rule vertically to separate different items.

With one or more frames selected,
script sets L of line = to the height of the frame
script sets Y of line = to Y of the frame

setting X coordinate (see InDesign document for examples):
——————————
a) If frame is on left hand page, and if frame is closer to the outside of the page than to the spine of the page, then script sets X of line = to right side of the frame and plus 0.0783 inches to the right
b) If frame is on left hand page, and if frame is closer to the spine of the page than the outside of the page, then script sets X of line = to left side of the frame and minus 0.0783 inches to the left
c) If frame is on right hand page, and if frame is closer to the spine of the page than the outside of the page, then script sets X of line = to right side of the frame and plus 0.0783 in to the right
d) If frame is on right hand page, and if frame is closer to the outside of the page than to the spine of the page, then script sets X of line = to left side of the frame and minus 0.0783 inches to the left
——————————

script applies 'cutoff rule' object style to the line.

With two frames selected,
script inserts line
script sets L of line = to the top of the highest frame to the bottom of the lowest frame
script sets Y of line = to the top of the frame that is highest of the two frames
script sets X of line = to between the two frames
script applies 'cutoff rule' object style to the line.

With one or more frames and a line selected,
script sets L of line = to the top of the highest frame to the bottom of the lowest frame
script sets Y of line = to the top of the frame that is highest of the two frames
script applies 'cutoff rule' object style to the line.
=======================================================================================================

Cutoff rule horizontal

The goal is automatically place a Cutoff Rule horizontally to separate different items.

With one frame selected,
script sets L of line = to the width of the frame
script sets X of line = to X of the frame
script sets Y of line = to Y of the frame plus 10 pts
script applies 'cutoff rule' object style to the line.

With two frames selected,
script inserts line
script sets L of line = to the left edge of the frame furthest to the left and to the right edge of the frame furthest to the right
script sets X of line = to the left edge of the frame furthest to the left
script sets Y of line = 10 pts above the top of the frame that is highest of the two frames
script applies 'cutoff rule' object style to the line.
=======================================================================================================

5 XII 2017
Off the top of my head, the cutoff rule script should be adjusted:
(horizontal) Place the line in between the two objects. Right now the script places the line in the middle of the two objects. 
It should look at the right edge of object A and left edge of object B and place the line between.

-- Fixed in version 1.1
=======================================================================================================

20 XII 2017
For the vertical cutoff rule, I was wondering if it's possible to have the script draw the line according to the top of one object to the bottom of another object, 
even if the objects are not the same width?

*/